//****************************************************************************************
// Name:		proc.cpp
// Platform:	SQL Server 2000 SP3a or higher, Windows NT, 2000 or XP
// Author:		Copyright (c) 2006 by Michael Coles, MCDBA
//
// Description:	Implements the xp_decrypt_aes XP.  Based on the Rijndael/AES 
//				implementation
//
// LEGAL STUFF:
// ------------
// Copyright (C) 2005 - 2006 by Michael Coles, MCDBA
//
// Some included code included is released under the redistribution agreements as 
// specified by the authors of the respective code.  Copyright holders of this included 
// code maintain all copyright and other rights to their original or derivative works.
//
// All rights reserved.                          
//
// REDISTRIBUTION OF THIS CODE:
// ----------------------------
// All code included in this package is either the original work of the copyright holder,
// or derivative work based on other copyright holders' works.  All derivative works 
// include information as required by the copright holders' redistribution agreements.
// These redistribution agreements, where possible, are included in the text of the source
// code distributed with this code.
//
// Redistribution and use in source and binary forms, with or without modification, are 
// permitted provided that the following conditions are met:
//
//   1. Redistributions of source code must retain the above copyright notice, this list 
//      of conditions and the following disclaimer.
//
//   2. Redistributions in binary form must reproduce the above copyright notice, this 
//      list of conditions and the following disclaimer in the documentation and/or other 
//      materials provided with the distribution.
//
//   3. The names of its contributors may not be used to endorse or promote products 
//      derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, 
// INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED 
// TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
// BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY 
// WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//****************************************************************************************
#include <dblib.h>
#include <stdafx.h>
#include <constants.h>
#include <params.h>
#include <aes.h>
#include <sha2.h>
#include <stdexcept>

#ifdef __cplusplus
extern "C" {
#endif

RETCODE __declspec(dllexport) xp_decrypt_aes(SRV_PROC *srvproc);

#ifdef __cplusplus
}
#endif

RETCODE __declspec(dllexport) xp_decrypt_aes(SRV_PROC *srvproc)
{
	RETCODE rc = XP_NOERROR;
	params *P1 = new params();	// encrypted text
	params *P2 = new params();	// plain text OUTPUT
	params *P3 = new params();	// password for master-key
	params *P4 = new params();	// master key
	params *P5 = new params();	// local key
	params *P6 = new params();	// key bit length

	BYTE password[128];
	ULONG passwordlength = 0;
	BYTE passhash[64];

	BYTE *localkey = new BYTE[80];
	int localkeylength = 0;
	BYTE *masterkey = new BYTE[64];
	int masterkeylength = 0;
	BYTE *localkeydecrypt = new BYTE[64];
	BYTE *encryptedtext = NULL;
	ULONG encryptedtextlength = 0;
	BYTE *plaintext = NULL;
	ULONG plaintextlength = 0;
	Rijndael::KeyLength keybits = Rijndael::Key32Bytes;

	try
	{
		// *** In This section we verify and grab all the parameters.  This is the majority
		// *** of the code.

		int numparams = params::getparamcount(srvproc);
		if (numparams != 6) { // This XP requires exactly six parameters
			Dblib::printerror(srvproc, USAGE_DECRYPT_AES);
			rc = XP_ERROR;
		}
		if (rc == XP_NOERROR) { // get all parameters
			params::getparam(srvproc, 1, P1);
			params::getparam(srvproc, 2, P2);
			params::getparam(srvproc, 3, P3);
			params::getparam(srvproc, 4, P4);
			params::getparam(srvproc, 5, P5);
			params::getparam(srvproc, 6, P6);
			if (P1->isoutput || !P2->isoutput || P3->isoutput || P4->isoutput || P5->isoutput || P6->isoutput) {
				Dblib::printerror(srvproc, USAGE_DECRYPT_AES);
				rc = XP_ERROR;
			} else if (P1->isnull || P4->isnull || P5->isnull) {
				rc = XP_ERROR;
			}
		}
		if (rc == XP_NOERROR) {
			if (P6->isnull) {
				keybits = Rijndael::Key32Bytes;
			} else {
				INT *i = (INT *)P6->cdata;
				switch (*i) {
					case 256:
						keybits = Rijndael::Key32Bytes;
						break;
					case 192:
						keybits = Rijndael::Key24Bytes;
						break;
					case 128:
						keybits = Rijndael::Key16Bytes;
						break;
					default:
						Dblib::printerror(srvproc, ERR_BAD_KEY_BITS);
						rc = XP_ERROR;
						break;
				}
			}
		}
		if (rc == XP_NOERROR) {
			if (P2->maxlength < P1->length) {
				Dblib::printerror(srvproc, ERR_OUTPUT_LENGTH);
				rc = XP_ERROR;
			}
		}
		if (rc == XP_NOERROR) {
			if (!P3->isnull) { // If password parameter is not NULL, grab the fourth param
				srv_bzero(password, 128);
				memcpy(password, P3->cdata, P3->length);
				passwordlength = P3->length;
			} else { // If fourth parameter is NULL
				srv_bzero(password, 128);
				strcpy((char *)password, srv_pfield(srvproc, SRV_USER, (int *)NULL));
				strupr((char *)password);
				passwordlength = (int)strlen((char *)password);
			}
		} else { // Otherwise, with 3 params, grab the user name and use it to decrypt the master key
			srv_bzero(password, 128);
			strcpy((char *)password, srv_pfield(srvproc, SRV_USER, (int *)NULL));
			strupr((char *)password);
			passwordlength = (int)strlen((char *)password);
		}
		if (rc == XP_NOERROR) {
			sha512_ctx ctx[1];	// Create an SHA-512 hash of the password
			sha512_begin(ctx);
			sha512_hash(password, passwordlength, ctx);
			sha512_end(passhash, ctx);
			// Get encrypted text
			encryptedtext = new BYTE[P1->length + 1];
			memcpy(encryptedtext, P1->cdata, P1->length);
			*(encryptedtext + P1->length) = '\0';
			encryptedtextlength = P1->length;
			// Get encrypted master key
			if (P4->length >= 64) {
				memcpy(masterkey, P4->cdata, 64);
				masterkeylength = 64;
			}
			if (P5->length >= 80) {
				memcpy(localkey, P5->cdata, 80);
				localkeylength = 80;
			}
			if (localkeylength == 0 || masterkeylength == 0) {
				Dblib::printerror(srvproc, ERR_KEY_LENGTH);
				rc = XP_ERROR;
			}
		}

		// *** Here we decrypt our master key, and use it to decrypt our local key.

		if (rc == XP_NOERROR) { // If we made it this far, we've grabbed and confirmed 
								// all of the parameters
			HCRYPTPROV hProv = NULL;
			if(!CryptAcquireContext(	// Grab a Crypto API Context
				&hProv,					// handle to the CSP
				SCRYPTO_KEY_CONTAINER,	// container name 
				NULL,					// use the default provider
				PROV_RSA_FULL,			// provider type
				CRYPT_MACHINE_KEYSET))	// flag values
			{
				Dblib::printerror(srvproc, ERR_CRYPTOAPI_CONTAINER);
				rc = XP_NOERROR;
			} else {
				DWORD dwSize = 64;
				if (rc == XP_NOERROR) {
					HCRYPTKEY hKey = NULL;
					HCRYPTHASH hHash = NULL;
					// Create a hash object.
					if (!CryptCreateHash(hProv, CALG_SHA1, 0, 0, &hHash)) {
						// Create a CryptoAPI hash object
						Dblib::printerror(srvproc, ERR_CRYPTOAPI_CREATEHASH);
						rc = XP_ERROR;
					}
					if (rc == XP_NOERROR && !CryptHashData(hHash, (CONST BYTE *)passhash, 64, 0)) {
						// Create a CryptoAPI hash of the password
						Dblib::printerror(srvproc, ERR_CRYPTOAPI_SHA1);
						rc = XP_ERROR;
					} 
					if (rc == XP_NOERROR && !CryptDeriveKey(hProv, CALG_RC4, hHash, 0, &hKey)) {
						// Derive a session key using the Crypto API
						Dblib::printerror(srvproc, ERR_CRYPTOAPI_DERIVE_KEY);
						rc = XP_ERROR;
					}
					if (rc == XP_NOERROR) {
						// Set variable to length of data in buffer.
						dwSize = 64;
						// Now decrypt data.
						if (!CryptDecrypt(			// Perform the actual decryption
									hKey,			// Key obtained earlier
									0,				// No hashing of data
									TRUE,			// Final or only buffer of data
									0,              // Must be zero
									masterkey,		// Data buffer
									&dwSize)) 		// Size of data
						{
							Dblib::printerror(srvproc, ERR_CRYPTOAPI_DEC_MASTER);
							rc = XP_ERROR;
						}
						// Clean up
						CryptDestroyKey(hKey);
						CryptDestroyHash(hHash);
						CryptReleaseContext(hProv, 0);
					}

					// *** Here we perform the encrypted text decryption using our local key

					if (rc == XP_NOERROR) {
						// Decrypting the local key by using the master key
						Rijndael rin;
						rin.init(Rijndael::CBC, Rijndael::Decrypt, masterkey, Rijndael::Key32Bytes);
						int len = rin.padDecrypt(localkey, 80, localkeydecrypt);
						// Finally, the show we all came to see -- here we actually decrypt
						// the encrypted text using the local key
						rin.init(Rijndael::CBC, Rijndael::Decrypt, localkeydecrypt, keybits);
						plaintext = new BYTE[encryptedtextlength + 16];
						plaintextlength = rin.padDecrypt(encryptedtext, encryptedtextlength, plaintext);
					}
				}
			}
		}
	} catch(std::exception ex) {
		Dblib::printerror(srvproc, ERR_CRYPT_EXCEPTION);
		rc = XP_ERROR;
	}
	// *** Return the result to SQL Server via the OUTPUT parameter
	if (rc == XP_NOERROR) {
		srv_paramsetoutput(srvproc, 2, plaintext, plaintextlength, FALSE);
	} else {
		srv_paramsetoutput(srvproc, 2, (BYTE *)"", 0, TRUE);
	}
	srv_senddone(srvproc, SRV_DONE_MORE, (DBUSMALLINT)0, (DBINT)1);
	// *** A little clean-up code
	if (encryptedtext != NULL) {
		srv_bzero(encryptedtext, encryptedtextlength);
		delete [] encryptedtext;
	}
	encryptedtext = NULL;
	if (plaintext != NULL) {
		srv_bzero(plaintext, plaintextlength);
		delete [] plaintext;
	}
	plaintext = NULL;
	if (localkey != NULL) {
		srv_bzero(localkey, 80);
		delete [] localkey;
	}
	localkey = NULL;
	if (masterkey != NULL) {
		srv_bzero(masterkey, 64);
		delete [] masterkey;
	}
	masterkey = NULL;
	if (localkeydecrypt != NULL)
	{
		srv_bzero(localkeydecrypt, 64);
		delete [] localkeydecrypt;
	}
	localkeydecrypt = NULL;
	if (P1 != NULL)
		delete P1;
	if (P2 != NULL)
		delete P2;
	if (P3 != NULL)
		delete P3;
	if (P4 != NULL)
		delete P4;
	if (P5 != NULL)
		delete P5;
	return rc;
}

